// $Id: SchedulerBasicP.nc,v 1.2 2010-06-29 22:07:52 scipio Exp $

/*									tab:4
 * Copyright (c) 2000-2003 The Regents of the University  of California.  
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * - Redistributions of source code must retain the above copyright
 *   notice, this list of conditions and the following disclaimer.
 * - Redistributions in binary form must reproduce the above copyright
 *   notice, this list of conditions and the following disclaimer in the
 *   documentation and/or other materials provided with the
 *   distribution.
 * - Neither the name of the University of California nor the names of
 *   its contributors may be used to endorse or promote products derived
 *   from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL
 * THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * Copyright (c) 2002-2003 Intel Corporation
 * All rights reserved.
 *
 * This file is distributed under the terms in the attached INTEL-LICENSE     
 * file. If you do not find these files, copies can be found by writing to
 * Intel Research Berkeley, 2150 Shattuck Avenue, Suite 1300, Berkeley, CA, 
 * 94704.  Attention:  Intel License Inquiry.
 */

/**
 * SchedulerBasicP implements the default TinyOS scheduler sequence, as
 * documented in TEP 106.
 *
 * @author Philip Levis
 * @author Cory Sharp
 * @author Kevin Klues <klueska@cs.stanford.edu>
 * @date   January 19 2005
 */

#include "hardware.h"
#include "thread.h"

module SchedulerBasicP {
  provides interface TaskScheduler;
  provides interface TaskBasic[uint8_t id];
  uses interface ThreadScheduler;
  uses interface Leds;
}
implementation {
  enum {
    NUM_TASKS = uniqueCount("TinyTaskSchedulerC.TaskBasic"),
    NO_TASK = 255,
  };

  volatile uint8_t m_head;
  volatile uint8_t m_tail;
  volatile uint8_t m_next[NUM_TASKS];

  // Helper functions (internal functions) intentionally do not have atomic
  // sections.  It is left as the duty of the exported interface functions to
  // manage atomicity to minimize chances for binary code bloat.

  // move the head forward
  // if the head is at the end, mark the tail at the end, too
  // mark the task as not in the queue
  inline uint8_t popTask() {
    if( m_head != NO_TASK ) {
      uint8_t id = m_head;
      m_head = m_next[m_head];
      if( m_head == NO_TASK ) {
	    m_tail = NO_TASK;
      }
      m_next[id] = NO_TASK;
      return id;
    }
    else {
      return NO_TASK;
    }
  }
  
  bool isWaiting( uint8_t id ) {
    return (m_next[id] != NO_TASK) || (m_tail == id);
  }
  
  async command bool TaskScheduler.hasTasks() {
    atomic return (m_head != NO_TASK);
  }

  bool pushTask( uint8_t id ) {
    if( !isWaiting(id) ) {
      if( m_head == NO_TASK ) {
	    m_head = id;
	    m_tail = id;
      }
      else {
	    m_next[m_tail] = id;
	    m_tail = id;
      }
      return TRUE;
    }
    else {
      return FALSE;
    }
  }
  
  command void TaskScheduler.init() {
    atomic {
      memset( (void *)m_next, NO_TASK, sizeof(m_next) );
      m_head = NO_TASK;
      m_tail = NO_TASK;
    }
  }
  
  command bool TaskScheduler.runNextTask() {
    uint8_t nextTask;
    atomic {
      nextTask = popTask();
      if( nextTask == NO_TASK ) {
	    return FALSE;
      }
    }
    signal TaskBasic.runTask[nextTask]();
    return TRUE;
  }

  command void TaskScheduler.taskLoop() {
    for (;;) {
      uint8_t nextTask;

      atomic {
	    while((nextTask = popTask()) == NO_TASK) {
	      call ThreadScheduler.suspendCurrentThread();
	    }
      }
      signal TaskBasic.runTask[nextTask]();
    }
  }

  /**
   * Return SUCCESS if the post succeeded, EBUSY if it was already posted.
   */
  
  async command error_t TaskBasic.postTask[uint8_t id]() {
    atomic { return pushTask(id) ? SUCCESS : EBUSY; }
  }

  default event void TaskBasic.runTask[uint8_t id]() {}
}

