#!/bin/sh
#
# gen-posix-names.sh
#
# COPYRIGHT (c) 1996 AT&T Research.
#
# Generate string-to-int tables for run-time POSIX values
# queried using sysconf and (f)pathconf.
#
# Usage: gen-posix-names.sh <prefix> <outfile>
#

# redefine PATH so that we get the right versions of the various tools
#
PATH=/bin:/usr/bin

# set locale variables so that sort works right
#
export LC_CTYPE LC_COLLATE
LC_CTYPE=C
LC_COLLATE=C

CPP=${CPP:-/lib/cpp}

PFIX=$1      # prefix: _SC_ or _PC_
OUTF=$2      # name of output file

INCLFILE="none"
USED_ENUMS="FALSE"

# target specific workarounds
#
case "$VERSION" in
  # linux uses enums for the _SC_ constants, so we cannot use the #ifdef check to avoid symbols
  # that are not really defined in unistd.h.
  *linux*)
      USED_ENUMS="TRUE"
      INCLFILE=tmp$$.h
      SRCFILE=tmp$$.c
      echo "#include <unistd.h>" > $SRCFILE
      $CPP $SRCFILE > $INCLFILE
      rm -f $SRCFILE
    ;;
  # newer versions of the Mac OS X developer tools keep the include files inside the Xcode
  # application bundle, so we add that as a possible path.
  *x86-darwin*)
    case `uname -r` in
      13.*) SDK=MacOSX10.9.sdk ;;
      *) SDK=none ;;
    esac
    if test x$SDK != xnone ; then
      # note: at some point, we might use "xcrun --show-sdk-path", but that only works
      # with Xcode 5.x+
      XCODE_DEV_PATH=`xcode-select --print-path`
      INCLFILE=$XCODE_DEV_PATH/Platforms/MacOSX.platform/Developer/SDKs/$SDK/usr/include/unistd.h
    fi
    ;;
  *) ;;
esac

if test x$INCLFILE = xnone ; then
# search the possible include files looking for a source
# of the constants.
  FILES="\
	/usr/include/unistd.h \
	/usr/include/sys/unistd.h \
	/usr/include/bsd/unistd.h \
	/usr/include/confname.h \
    "
  for f in $FILES ; do
    if test -r $f ; then
      grep $PFIX $f > /dev/null
      if test $? -eq 0 ; then
	INCLFILE=$f
	break;
      fi
    fi
  done
  if test x$INCLFILE = xnone ; then
    echo "gen-posix-names.sh: unable to find <unistd.h>"
    exit 1
  fi
fi

echo "/* $OUTF" >> $OUTF
echo " *"       >> $OUTF
echo " * This file is generated by gen-posix-names.sh from $INCLFILE" >> $OUTF
echo " */"       >> $OUTF

if [ "$USED_ENUMS" = "TRUE" ]; then
  for i in `sed -n "s/.*$PFIX\([0-9A-Z_]*\).*/\1/p" $INCLFILE | sort -u`
  do
    echo "  {\"$i\",  $PFIX$i}," >> $OUTF
  done
else
  for i in `sed -n "s/.*$PFIX\([0-9A-Z_]*\).*/\1/p" $INCLFILE | sort -u`
  do
    echo "#ifdef $PFIX$i" >> $OUTF
    echo "  {\"$i\",  $PFIX$i}," >> $OUTF
    echo "#endif" >> $OUTF
  done
fi

if [ "$USED_ENUMS" = "TRUE" ]; then
  rm -f $INCLFILE
fi

exit 0
