/*******************************************************************
 * Fritz Fun                                                       *
 * Created by Jan-Michael Brummer                                  *
 * All parts are distributed under the terms of GPLv2. See COPYING *
 *******************************************************************/

/**
 * \file dialnumber.c
 * \brief Dial number dialog
 */

#include <ffgtk.h>
#include <faxophone/isdn-convert.h>

#include <llevel.h>

enum {
	CONTACT_NAME,
	CONTACT_NUMBER
};

/** Current dial number */
static gchar *pnDialNumber = NULL;
/** Current dial port */
static gchar *pnCurrentPort = NULL;
/** Current dialing dialog */
static GtkWidget *psDialingDialog = NULL;
/** Current ok button */
static GtkWidget *psOkButton = NULL;
/** Current cancel button */
static GtkWidget *psCancelButton = NULL;

static GtkWidget *psLineLevelIn = NULL;
static GtkWidget *psLineLevelOut = NULL;

static guint nTimer = 0;
static time_t nConnectTime;
static GtkWidget *psTimeLabel = NULL;

/**
 * \brief Set buttons according session type
 * \param nType session type
 */
static void setButtons( gint nType ) {
	if ( psCancelButton != NULL ) {
		gtk_widget_destroy( psCancelButton );
	}
	if ( psOkButton != NULL ) {
		gtk_widget_destroy( psOkButton );
	}

	if ( nType == 0 ) {
		/* Add cancel button and enable it */
		psCancelButton = gtk_dialog_add_button( GTK_DIALOG( psDialingDialog ), _( "Hangup" ), GTK_RESPONSE_CANCEL );
		gtk_widget_set_sensitive( psCancelButton, TRUE );

		/* Add ok button and disable it */
		psOkButton = gtk_dialog_add_button( GTK_DIALOG( psDialingDialog ), GTK_STOCK_CLOSE, GTK_RESPONSE_OK );
		gtk_widget_set_sensitive( psOkButton, FALSE );
	} else {
		/* Add ok button and enable it */
		psOkButton = gtk_dialog_add_button( GTK_DIALOG( psDialingDialog ), GTK_STOCK_CLOSE, GTK_RESPONSE_OK );
		gtk_widget_set_sensitive( psOkButton, TRUE );

		/* Add redo button and enable it */
		psCancelButton = gtk_dialog_add_button( GTK_DIALOG( psDialingDialog ),_( "Redial" ), 1 );
		gtk_widget_set_sensitive( psCancelButton, TRUE );
	}
}

/**
 * \brief Set dial number, needed for call monitor match
 * \param pnNumber fix number we want to called
 */
void dialSetNumber( const gchar *pnNumber ) {
	if ( pnDialNumber != NULL ) {
		g_free( pnDialNumber );
		pnDialNumber = NULL;
	}

	if ( pnNumber != NULL ) {
		pnDialNumber = g_strdup( pnNumber );
	}
}

/**
 * \brief Get current dial number
 * \return current fix dial number
 */
gchar *dialGetNumber( void ) {
	return pnDialNumber;
}

/**
 * \brief Get current dial dialog
 * \return dialog widget
 */
GtkWidget *dialGetDialog( void ) {
	return psDialingDialog;
}

/**
 * \brief Set dial number (called from AddressBook)
 * \param psParent parent widget containing name and number entry
 * \param pnName name
 * \param pnNumber fax number
 */
void setDialNameNumber( GtkWidget *psParent, const gchar *pnName, const gchar *pnNumber ) {
	GtkWidget *psNameEntry = g_object_get_data( G_OBJECT( psParent ), "name_entry" );
	GtkWidget *psNumberEntry = g_object_get_data( G_OBJECT( psParent ), "number_entry" );

	if ( psNumberEntry != NULL && pnNumber != NULL ) {
		gtk_entry_set_text( GTK_ENTRY( psNumberEntry ), pnNumber );
	}

	if ( psNameEntry != NULL && pnName != NULL ) {
		gtk_entry_set_text( GTK_ENTRY( psNameEntry ), pnName );
	}
}

/**
 * \brief Create elapsed time string from two times
 * \param time1 first time info
 * \param time0 second time info
 * \return time string
 */
static char *timediff_str( time_t time1, time_t time0 ) {
	int nDuration;
	char *pnBuf;
	int nHours;
	int nMinutes;
	int nSeconds;

	pnBuf = ( char * ) malloc( 9 );

	nDuration = ( int ) difftime( time1, time0 );
	nSeconds = nDuration % 60;
	nMinutes = ( nDuration / 60 ) % 60;
	nHours = nDuration / ( 60 * 60 );

	if ( snprintf( pnBuf, 9, "%02d:%02d:%02d", nHours, nMinutes, nSeconds ) >= 9 ) {
		pnBuf[ 8 ] = '\0';
	}

	return pnBuf;
}

/**
 * \brief Session time which updates line level and time status
 * \param pData UNUSED
 * \return TRUE
 */
static gboolean sessionTimer( gpointer pData ) {
	GString *psBuf;
	char *pnTimeDiff = timediff_str( time( NULL ), nConnectTime );

	psBuf = g_string_new( "" );
	if ( psBuf != NULL ) {
		g_string_append_printf( psBuf, _( "Time: %s" ), pnTimeDiff );

		gtk_label_set_text( GTK_LABEL( psTimeLabel ), psBuf -> str );
		g_string_free( psBuf, TRUE );
	}
	free( pnTimeDiff );

#ifdef HAVE_FAXOPHONE
	lineLevelSet( psLineLevelIn, log10( 1.0 + 9.0 * getLineLevelIn() ) );
	lineLevelSet( psLineLevelOut, log10( 1.0 + 9.0 * getLineLevelOut() ) );

	/* Flush recording buffer (if needed) */
	softphoneFlush();
#endif

	return TRUE;
}

/**
 * \brief Set status label in dial window
 * \param psDialog dial dialog window
 * \param nType call type
 */
void dialWindowSetStatus( GtkWidget *psDialog, gint nType ) {
	if ( psDialog != NULL ) {
		GtkWidget *psLabel = g_object_get_data( G_OBJECT( psDialog ), "status_label" );

		if ( psLabel != NULL ) {
			gchar *pnTmp = NULL;

			switch ( nType ) {
				case CALL_TYPE_DISCONNECT:
					pnTmp = g_strdup( _( "Status: Disconnect" ) );
					setButtons( 1 );
					if ( nTimer != 0 ) {
						g_source_remove( nTimer );
						nTimer = 0;
						lineLevelSet( psLineLevelIn, 0 );
						lineLevelSet( psLineLevelOut, 0 );
					}
					break;
				case CALL_TYPE_CONNECT:
					pnTmp = g_strdup( _( "Status: Connect" ) );
					nConnectTime = time( NULL );
					if ( nTimer == 0 ) {
						nTimer = g_timeout_add( 200, sessionTimer, NULL );
					}
					Debug( KERN_DEBUG, "Timer is %d\n", nTimer );
					break;
				case 42:
					pnTmp = g_strdup( _( "Status: Dial" ) );
					break;
			}

			gtk_label_set_text( GTK_LABEL( psLabel ), pnTmp );

			g_free( pnTmp );
		}
	}
}

/**
 * \brief Dialing window response callback
 * \param psDialog dialog window pointer
 * \param nResponse response id
 * \param pUserData file name
 */
static void dialingResponse( GtkDialog *psDialog, gint nResponse, gpointer pUserData ) {
	char *pnPort = g_object_get_data( G_OBJECT( psDialog ), "port_name" );
	gint nError = -1;

	if ( nTimer != 0 ) {
		g_source_remove( nTimer );
		nTimer = 0;
		lineLevelSet( psLineLevelIn, 0 );
		lineLevelSet( psLineLevelOut, 0 );
	}

	Debug( KERN_DEBUG, "Called with response: %d\n", nResponse );
	switch ( nResponse ) {
		case 1:
			/* Dial */
			Debug( KERN_DEBUG, "Dial...%s\n", pnPort );
#ifdef HAVE_FAXOPHONE
			if ( strcmp( pnPort, _( "Softphone" ) ) == 0 ) {
				/* TODO: Fix anonymous setting */
				softphoneDial( pnDialNumber, FALSE );
				nError = 0;
			} else
#endif
			{
				nError = routerCallNumber( getActiveProfile(), pnDialNumber, pnCurrentPort );
			}

			if ( nError != 0 ) {
				if ( nError == 33485 ) {
					ShowError( FALSE, _( "Login blocked" ), _( "ffgtk has blocked login until valid password is given" ) );
				} else if ( nError < 0 ) {
					RouterLoginError( FALSE, -nError );
				} else {
					ShowError( FALSE, _( "Network error" ), curl_easy_strerror( nError ) );
				}
			} else {
				Debug( KERN_DEBUG, "Set status\n" );
				dialWindowSetStatus( psDialingDialog, 42 );
				Debug( KERN_DEBUG, "Set buttons\n" );
				setButtons( 0 );
				Debug( KERN_DEBUG, "return\n" );
				return;
			}
			break;
		case GTK_RESPONSE_CANCEL:
			Debug( KERN_DEBUG, "RESPONSE_CANCEL\n" );
#ifdef HAVE_FAXOPHONE
			if ( strcmp( pnPort, _( "Softphone" ) ) == 0 ) {
				softphoneHangup();
				dialWindowSetStatus( psDialingDialog, 42 );
				setButtons( 0 );
				return;
			} else
#endif
			{
				nError = routerHangup( getActiveProfile(), pnPort );
			}
			if ( nError != 0 ) {
				if ( nError == 33485 ) {
					ShowError( FALSE, _( "Login blocked" ), _( "ffgtk has blocked login until valid password is given" ) );
				} else if ( nError < 0 ) {
					RouterLoginError( FALSE, -nError );
				} else {
					ShowError( FALSE, _( "Network error" ), curl_easy_strerror( nError ) );
				}
			}
			break;
		case GTK_RESPONSE_DELETE_EVENT:
			Debug( KERN_DEBUG, "DELETE_EVENT\n" );
			break;
	}
	dialSetNumber( NULL );

	if ( pnCurrentPort != NULL ) {
		g_free( pnCurrentPort );
		pnCurrentPort = NULL;
	}

	Debug( KERN_DEBUG, "Destroy\n" );
	gtk_widget_destroy( GTK_WIDGET( psDialog ) );
	psDialingDialog = NULL;
}

#ifdef HAVE_FAXOPHONE
/**
 * \brief DTMF callback
 * \param psWidget button widget
 * \param pUserData dtmf code
 */
static void dtmfClicked( GtkWidget *psWidget, gpointer pUserData ) {
	gint nNum = GPOINTER_TO_INT( pUserData );

	softphoneSendDtmfCode( nNum );
}

/**
 * \brief Mute callback
 * \param psWidget button widget
 * \param pUserData UNUSED
 */
static void muteClicked( GtkWidget *psWidget, gpointer pUserData ) {
	static int nMute = 0;

	nMute = !nMute;
	softphoneMute( nMute );
	gtk_toggle_button_set_mode( GTK_TOGGLE_BUTTON( psWidget ), nMute );
}

/**
 * \brief Hold callback
 * \param psWidget button widget
 * \param pUserData UNUSED
 */
static void holdClicked( GtkWidget *psWidget, gpointer pUserData ) {
	static int nHold = 0;

	nHold = !nHold;
	softphoneHold( nHold );
}

/**
 * \brief Record callback
 * \param psWidget button widget
 * \param pUserData UNUSED
 */
static void recordToggled( GtkWidget *psWidget, gpointer pUserData ) {
	GtkToggleButton *psButton = GTK_TOGGLE_BUTTON( psWidget );
	gboolean bState = gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( psButton ) );

	softphoneRecord( bState );
}
#endif

/**
 * \brief Dialing window
 * \param pnNumber dial number
 * \param pnPort port name
 */
void dialingDialog( const gchar *pnNumber, const gchar *pnPort ) {
	GtkBuilder *psBuilder = NULL;
	GError *psError = NULL;
	GtkWidget *psNameLabel = NULL;
	GtkWidget *psNumberLabel = NULL;
	GtkWidget *psStatusLabel = NULL;
	GtkWidget *psImage = NULL;
	GtkWidget *psOptions = NULL;
#ifdef HAVE_FAXOPHONE
	GtkWidget *psButton1;
	GtkWidget *psButton2;
	GtkWidget *psButton3;
	GtkWidget *psButton4;
	GtkWidget *psButton5;
	GtkWidget *psButton6;
	GtkWidget *psButton7;
	GtkWidget *psButton8;
	GtkWidget *psButton9;
	GtkWidget *psButton0;
	GtkWidget *psButtonAsterisk;
	GtkWidget *psButtonPound;
	GtkWidget *psHoldButton;
	GtkWidget *psMuteButton;
	GtkWidget *psRecordCheckButton;
	GtkWidget *psLineLevelBox;
	GtkWidget *psBox;
#endif
	gchar *pnTmp = NULL;
	gchar *pnUiFile;
	struct sPerson *psPerson = NULL;
	GdkPixbuf *psScaled = NULL;

	psBuilder = gtk_builder_new();
	pnUiFile = getUiFile( "dialing.ui" );
	if ( gtk_builder_add_from_file( psBuilder, pnUiFile, &psError ) == 0 ) {
	    Debug( KERN_WARNING, "Error: %s\n", psError -> message );
	    g_error_free( psError );
		g_free( pnUiFile );
	    return;
	}
	g_free( pnUiFile );

	psDialingDialog = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psDialingDialog" ) );

	psCancelButton = NULL;
	psOkButton = NULL;
	setButtons( 0 );

	g_object_set_data( G_OBJECT( psDialingDialog ), "port_name", ( gchar * ) pnPort );
	psNameLabel = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psNameLabel" ) );
	psNumberLabel = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psNumberLabel" ) );
	psImage = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psImage" ) );
	psTimeLabel = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psTimeLabel" ) );
	psStatusLabel = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psStatusLabel" ) );
	psOptions = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psExpander" ) );
	g_object_set_data( G_OBJECT( psDialingDialog ), "status_label", psStatusLabel );

#ifdef HAVE_FAXOPHONE
	psButton1 = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psButton1" ) );
	psButton2 = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psButton2" ) );
	psButton3 = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psButton3" ) );
	psButton4 = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psButton4" ) );
	psButton5 = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psButton5" ) );
	psButton6 = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psButton6" ) );
	psButton7 = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psButton7" ) );
	psButton8 = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psButton8" ) );
	psButton9 = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psButton9" ) );
	psButton0 = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psButton0" ) );
	psButtonAsterisk = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psButtonAsterisk" ) );
	psButtonPound = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psButtonPound" ) );

	psHoldButton = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psHoldButton" ) );
	psMuteButton = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psMuteButton" ) );
	psRecordCheckButton = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psRecordCheckButton" ) );

	psBox = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psBox" ) );

	psLineLevelBox = gtk_vbox_new( FALSE, 0 );
	gtk_box_pack_start( GTK_BOX( psBox ), psLineLevelBox, FALSE, FALSE, 0 );

	psLineLevelIn = lineLevelBarNew( 66, 10 );
	gtk_box_pack_start( GTK_BOX( psLineLevelBox ), psLineLevelIn, FALSE, FALSE, 0 );

	psLineLevelOut = lineLevelBarNew( 66, 10 );
	gtk_box_pack_start( GTK_BOX( psLineLevelBox ), psLineLevelOut, FALSE, FALSE, 0 );
	gtk_widget_show_all( psLineLevelBox );

	g_signal_connect( G_OBJECT( psButton1 ), "clicked", G_CALLBACK( dtmfClicked ), GINT_TO_POINTER( '1' ) );

	g_signal_connect( G_OBJECT( psButton2 ), "clicked", G_CALLBACK( dtmfClicked ), GINT_TO_POINTER( '2' ) );
	g_signal_connect( G_OBJECT( psButton3 ), "clicked", G_CALLBACK( dtmfClicked ), GINT_TO_POINTER( '3' ) );
	g_signal_connect( G_OBJECT( psButton4 ), "clicked", G_CALLBACK( dtmfClicked ), GINT_TO_POINTER( '4' ) );
	g_signal_connect( G_OBJECT( psButton5 ), "clicked", G_CALLBACK( dtmfClicked ), GINT_TO_POINTER( '5' ) );
	g_signal_connect( G_OBJECT( psButton6 ), "clicked", G_CALLBACK( dtmfClicked ), GINT_TO_POINTER( '6' ) );
	g_signal_connect( G_OBJECT( psButton7 ), "clicked", G_CALLBACK( dtmfClicked ), GINT_TO_POINTER( '7' ) );
	g_signal_connect( G_OBJECT( psButton8 ), "clicked", G_CALLBACK( dtmfClicked ), GINT_TO_POINTER( '8' ) );
	g_signal_connect( G_OBJECT( psButton9 ), "clicked", G_CALLBACK( dtmfClicked ), GINT_TO_POINTER( '9' ) );
	g_signal_connect( G_OBJECT( psButton0 ), "clicked", G_CALLBACK( dtmfClicked ), GINT_TO_POINTER( '0' ) );
	g_signal_connect( G_OBJECT( psButtonAsterisk ), "clicked", G_CALLBACK( dtmfClicked ), GINT_TO_POINTER( '*' ) );
	g_signal_connect( G_OBJECT( psButtonPound ), "clicked", G_CALLBACK( dtmfClicked ), GINT_TO_POINTER( '#' ) );

	g_signal_connect( G_OBJECT( psMuteButton ), "clicked", G_CALLBACK( muteClicked ), NULL );
	g_signal_connect( G_OBJECT( psHoldButton ), "clicked", G_CALLBACK( holdClicked ), NULL );
	g_signal_connect( G_OBJECT( psRecordCheckButton ), "toggled", G_CALLBACK( recordToggled ), NULL );
	
	if ( pnPort != NULL && !strcmp( pnPort, _( "Softphone" ) ) ) {
		gtk_expander_set_expanded( GTK_EXPANDER( psOptions ), TRUE );
		gtk_widget_show( GTK_WIDGET( psOptions ) );
	}
#else
	gtk_expander_set_expanded( GTK_EXPANDER( psOptions ), FALSE );
	gtk_widget_hide( GTK_WIDGET( psOptions ) );
#endif

	psPerson = findPersonByNumber( ( gchar * ) pnNumber );
	pnTmp = g_strdup_printf( "<b>%s</b>", psPerson ? psPerson -> pnDisplayName : _( "Unknown" ) );
	gtk_label_set_markup( GTK_LABEL( psNameLabel ), pnTmp );
	g_free( pnTmp );
	pnTmp = g_strdup_printf( _( "Number: %s" ), pnNumber );
	gtk_label_set_markup( GTK_LABEL( psNumberLabel ), pnTmp );
	g_free( pnTmp );

	psScaled = getScaledImage( psPerson, 2 );
	if ( psScaled != NULL ) {
		gtk_image_set_from_pixbuf( GTK_IMAGE( psImage ), psScaled );
	}

	gtk_builder_connect_signals( psBuilder, NULL );

	g_object_unref( G_OBJECT( psBuilder ) );

	g_signal_connect( G_OBJECT( psDialingDialog ), "response", G_CALLBACK( dialingResponse ), NULL );

	gtk_widget_show_all( psDialingDialog );
}

/**
 * \brief Set dial number
 * \param psItem menu item
 * \param pUserData dial number
 */
static void setDialNumber( GtkMenuItem *psItem, gpointer pUserData ) {
	GtkWidget *psEntry = g_object_get_data( G_OBJECT( psItem ), "entry" );

	gtk_entry_set_text( GTK_ENTRY( psEntry ), pUserData );
}

/**
 * \brief Set popup menu below name widget
 * \param psMenu menu we want to set
 * \param pnX where to store the x value
 * \param pnY where to store the y value
 * \param pbPushIn allow gtk to push menu into the visible screen
 * \param pUserData contains number widget where we get the position from
 */
static void menuSetPosition( GtkMenu *psMenu, gint *pnX, gint *pnY, gboolean *pbPushIn, gpointer pUserData ) {
	GtkWidget *psWidget = GTK_WIDGET( pUserData );
	GtkRequisition sReq;

	gdk_window_get_origin( gtk_widget_get_window( psWidget ), pnX, pnY );
#if GTK_MAJOR_VERSION < 3
	gtk_widget_size_request( GTK_WIDGET( psMenu ), &sReq );
#else
	gtk_widget_get_preferred_size( GTK_WIDGET( psMenu ), &sReq, NULL );
	*pnX += 85;
	*pnY -= 12;
#endif
	*pnY += sReq.height;
	*pbPushIn = TRUE;
}

/**
 * \brief Try to match name with addressbook
 * \param psWidget entry completion widget
 * \param psModel tree model
 * \param psIter tree iterator
 * \param pUserData number entry
 * \return FALSE
 */
gboolean dialNameMatchSelect( GtkEntryCompletion *psWidget, GtkTreeModel *psModel, GtkTreeIter *psIter, gpointer pUserData ) {  
	GtkWidget *psMenu = NULL;
	GtkWidget *psItem;
	GValue sValue = { 0, };
	struct sPerson *psPerson;
	gchar *pnTmp;

	gtk_tree_model_get_value( psModel, psIter, CONTACT_NAME, &sValue );
	psPerson = findPerson( g_value_get_string( &sValue ) );
	g_value_unset( &sValue );

	psMenu = gtk_menu_new();
	if ( psPerson -> pnBusinessPhone != NULL && strlen( psPerson -> pnBusinessPhone ) ) {
		pnTmp = g_strdup_printf( "WORK: %s", psPerson -> pnBusinessPhone );
		psItem = gtk_menu_item_new_with_label( pnTmp );
		gtk_menu_shell_append( GTK_MENU_SHELL( psMenu ), psItem );
		g_signal_connect( G_OBJECT( psItem ), "activate", G_CALLBACK( setDialNumber ), psPerson -> pnBusinessPhone );
		g_object_set_data( G_OBJECT( psItem ), "entry", pUserData );
		g_free( pnTmp );
	}
	if ( psPerson -> pnBusinessFax != NULL && strlen( psPerson -> pnBusinessFax ) ) {
		pnTmp = g_strdup_printf( "BUSINESS FAX: %s", psPerson -> pnBusinessFax );
		psItem = gtk_menu_item_new_with_label( pnTmp );
		gtk_menu_shell_append( GTK_MENU_SHELL( psMenu ), psItem );
		g_signal_connect( G_OBJECT( psItem ), "activate", G_CALLBACK( setDialNumber ), psPerson -> pnBusinessFax );
		g_object_set_data( G_OBJECT( psItem ), "entry", pUserData );
		g_free( pnTmp );
	}
	if ( psPerson -> pnPrivateMobile != NULL && strlen( psPerson -> pnPrivateMobile ) ) {
		pnTmp = g_strdup_printf( "MOBILE: %s", psPerson -> pnPrivateMobile );
		psItem = gtk_menu_item_new_with_label( pnTmp );
		gtk_menu_shell_append( GTK_MENU_SHELL( psMenu ), psItem );
		g_signal_connect( G_OBJECT( psItem ), "activate", G_CALLBACK( setDialNumber ), psPerson -> pnPrivateMobile );
		g_object_set_data( G_OBJECT( psItem ), "entry", pUserData );
		g_free( pnTmp );
	}
	if ( psPerson -> pnPrivatePhone != NULL && strlen( psPerson -> pnPrivatePhone ) ) {
		pnTmp = g_strdup_printf( "HOME: %s", psPerson -> pnPrivatePhone );
		psItem = gtk_menu_item_new_with_label( pnTmp );
		gtk_menu_shell_append( GTK_MENU_SHELL( psMenu ), psItem );
		g_signal_connect( G_OBJECT( psItem ), "activate", G_CALLBACK( setDialNumber ), psPerson -> pnPrivatePhone );
		g_object_set_data( G_OBJECT( psItem ), "entry", pUserData );
		g_free( pnTmp );
	}
	if ( psPerson -> pnPrivateFax != NULL && strlen( psPerson -> pnPrivateFax ) ) {
		pnTmp = g_strdup_printf( "PRIVATE FAX: %s", psPerson -> pnPrivateFax );
		psItem = gtk_menu_item_new_with_label( pnTmp );
		gtk_menu_shell_append( GTK_MENU_SHELL( psMenu ), psItem );
		g_signal_connect( G_OBJECT( psItem ), "activate", G_CALLBACK( setDialNumber ), psPerson -> pnPrivateFax );
		g_object_set_data( G_OBJECT( psItem ), "entry", pUserData );
		g_free( pnTmp );
	}

	usleep( 200000 );
	gtk_widget_show_all( GTK_WIDGET( psMenu ) );
	gtk_menu_popup( GTK_MENU( psMenu ), NULL, NULL, menuSetPosition, gtk_entry_completion_get_entry( psWidget ), 0, gtk_get_current_event_time() );

	return FALSE;
}

/**
 * \brief Dial number response callback
 * \param psDialog dialog window
 * \param nResult response id
 * \param pData file name
 */
static void dialNumberResponse( GtkDialog *psDialog, gint nResult, gpointer pData ) {
	GtkWidget *psEntry = g_object_get_data( G_OBJECT( psDialog ), "number_entry" );
	GtkWidget *psToggle = g_object_get_data( G_OBJECT( psDialog ), "suppress_checkbutton" );
	GtkWidget *psPort = g_object_get_data( G_OBJECT( psDialog ), "port_box" );
	gchar *pnPort = NULL;
	gchar *pnFixNumber;
	gint nError = -1;

	if ( nResult == 2 ) {
		gchar *pnNumber;

		if ( gtk_entry_get_text( GTK_ENTRY( psEntry ) ) == NULL ) {
			return;
		}

		pnNumber = g_strdup( gtk_entry_get_text( GTK_ENTRY( psEntry ) ) );
		pnFixNumber = fixNumber( pnNumber );
		g_free( pnNumber );
		pnPort = gtk_combo_box_get_active_text( GTK_COMBO_BOX( psPort ) );

		if ( gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( psToggle ) ) ) {
			pnNumber = g_strdup_printf( "*31#%s", pnFixNumber );
		} else {
			pnNumber = pnFixNumber;
		}

		/* Close previous dialing dialog */
		if ( psDialingDialog != NULL ) {
			gtk_widget_destroy( GTK_WIDGET( psDialingDialog ) );
			psDialingDialog = NULL;
		}

		/* Dial */
#ifdef HAVE_FAXOPHONE
		if ( strcmp( pnPort, _( "Softphone" ) ) == 0 ) {
			nError = softphoneDial( pnFixNumber, gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( psToggle ) ) );
		} else
#endif
		{
			nError = routerCallNumber( getActiveProfile(), pnNumber, pnPort );
		}
		if ( nError != 0 ) {
			if ( nError == 33485 ) {
				ShowError( FALSE, _( "Login blocked" ), _( "ffgtk has blocked login until valid password is given" ) );
			} else if ( nError < 0 ) {
				RouterLoginError( FALSE, -nError );
			} else {
				ShowError( FALSE, _( "Network error" ), curl_easy_strerror( nError ) );
			}
		} else {
			pnCurrentPort = g_strdup( pnPort );

			dialSetNumber( pnFixNumber );
			dialingDialog( pnFixNumber, pnPort );
		}

		g_free( pnNumber );
	}

	gtk_widget_destroy( GTK_WIDGET( psDialog ) );
}

/**
 * \brief Open addressbook in dial mode
 * \param psWidget button widget
 * \param pUserData UNUSED
 */
static void dialAddressBook( GtkWidget *psWidget, gpointer pUserData ) {
	GtkWidget *psDialog = AddressBook( 1 );

	g_object_set_data( G_OBJECT( psDialog ), "parent", pUserData );
}

/**
 * \brief Dial number callback
 * \param psPerson person structure
 * \param pnNumber number to dial
 */
void dialNumberDialog( struct sPerson *psPerson, const gchar *pnNumber ) {
	GtkBuilder *psBuilder = NULL;
	GError *psError = NULL;
	GtkWidget *psDialog = NULL;
	GList *psList = psPersonsList;
	GtkListStore *psModel = NULL;
	GtkTreeIter sIter;
	GtkEntryCompletion *psCompletion = NULL;
	GtkWidget *psToEntry = NULL;
	GtkWidget *psNumberEntry = NULL;
	GtkWidget *psPortBox = NULL;
	GtkWidget *psSuppressCheckButton = NULL;
	GtkWidget *psAddressBookButton = NULL;
	gchar *pnUiFile;

	psBuilder = gtk_builder_new();
	pnUiFile = getUiFile( "dialwindow.ui" );
	if ( gtk_builder_add_from_file( psBuilder, pnUiFile, &psError ) == 0 ) {
	    Debug( KERN_WARNING, "Error: %s\n", psError -> message );
	    g_error_free( psError );
		g_free( pnUiFile );
	    return;
	}
	g_free( pnUiFile );

	psDialog = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psDialDialog" ) );
	psToEntry = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psToEntry" ) );
	g_object_set_data( G_OBJECT( psDialog ), "name_entry", psToEntry );
	psNumberEntry = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psNumberEntry" ) );
	g_object_set_data( G_OBJECT( psDialog ), "number_entry", psNumberEntry );

	if ( psPerson != NULL ) {
		gtk_entry_set_text( GTK_ENTRY( psToEntry ), psPerson -> pnDisplayName );
	}

	if ( pnNumber != NULL ) {
		gtk_entry_set_text( GTK_ENTRY( psNumberEntry ), pnNumber );
	}

	psSuppressCheckButton = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psSuppressCheckButton" ) );
	g_object_set_data( G_OBJECT( psDialog ), "suppress_checkbutton", psSuppressCheckButton );
	psAddressBookButton = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psAddressBookButton" ) );
	g_signal_connect( G_OBJECT( psAddressBookButton ), "clicked", G_CALLBACK( dialAddressBook ), psDialog );
	psPortBox = GTK_WIDGET( gtk_builder_get_object( psBuilder, "psPortComboBox" ) );
	g_object_set_data( G_OBJECT( psDialog ), "port_box", psPortBox );
	routerAddComboBoxPorts( getActiveProfile(), psPortBox );

	/* Entry completion */
	psCompletion = gtk_entry_completion_new();
	gtk_entry_completion_set_text_column( psCompletion, CONTACT_NAME );
	psModel = gtk_list_store_new( 1, G_TYPE_STRING );

	while ( psList != NULL ) {
		struct sPerson *psPerson = psList -> data;

		if ( psPerson != NULL ) {
			gtk_list_store_append( psModel, &sIter );
			gtk_list_store_set( psModel, &sIter, CONTACT_NAME, psPerson -> pnDisplayName, -1 ); 
		}

		psList = psList -> next;
	}
	gtk_entry_completion_set_model( psCompletion, GTK_TREE_MODEL( psModel ) );
	gtk_entry_set_completion( GTK_ENTRY( psToEntry ), psCompletion );
	g_signal_connect( G_OBJECT( psCompletion ), "match-selected", G_CALLBACK( dialNameMatchSelect ), psNumberEntry );

	gtk_builder_connect_signals( psBuilder, NULL );

	g_object_unref( G_OBJECT( psBuilder ) );

	g_signal_connect( G_OBJECT( psDialog ), "close", G_CALLBACK( gtk_widget_destroy ), psDialog );
	g_signal_connect( G_OBJECT( psDialog ), "response", G_CALLBACK( dialNumberResponse ), NULL );

	gtk_widget_show_all( psDialog );

	gtk_window_activate_focus( GTK_WINDOW( psDialog ) );
	Debug( KERN_DEBUG, "focus: %d\n", gtk_window_has_toplevel_focus( GTK_WINDOW( psDialog ) ) );
}
