#!/usr/bin/env python
# Copyright (c) 2012 The Chromium Authors. All rights reserved.
# Use of this source code is governed by a BSD-style license that can be
# found in the LICENSE file.

"""Generate a C++ header from input_methods.txt.

This program generates a C++ header file containing the information on login XKB
layout IDs. It parses input_methods.txt, and then generates a static array
definition from the information extracted. The input and output file names are
specified on the command line.

Run it like:
  gen_input_methods.py input_methods.txt input_methods.h

It will produce output that looks like:

// Automatically generated by gen_input_methods.py
#ifndef CHROMEOS_IME_INPUT_METHODS_H_
#define CHROMEOS_IME_INPUT_METHODS_H_

namespace chromeos {
namespace input_method {

const char* const kLoginXkbLayoutIds[] = {
  "us",
  "us(dvorak)",
  "be",
  "br",
};

}  // namespace input_method
}  // namespace chromeos

#endif  // CHROMEOS_IME_INPUT_METHODS_H_

"""

import fileinput
import re
import sys

OUTPUT_HEADER = """// Automatically generated by gen_input_methods.py
#ifndef CHROMEOS_IME_INPUT_METHODS_H_
#define CHROMEOS_IME_INPUT_METHODS_H_

namespace chromeos {
namespace input_method {

const char* const kLoginXkbLayoutIds[] = {
"""
ENGINE_FORMAT = ('  "%s",\n')
OUTPUT_FOOTER = """
};

}  // namespace input_method
}  // namespace chromeos

#endif  // CHROMEOS_IME_INPUT_METHODS_H_
"""

def CreateEngineHeader(login_xkb_layout_ids):
  """Create the header file from a list of login XKB layout IDs.

  Arguments:
    login_xkb_layout_ids: list of login XKB layout IDs
  Returns:
    The text of a C++ header file containing the login XKB layout IDs.
  """
  output = []
  output.append(OUTPUT_HEADER)
  for login_xkb_layout_id in login_xkb_layout_ids:
    output.append(ENGINE_FORMAT % login_xkb_layout_id)
  output.append(OUTPUT_FOOTER)

  return "".join(output)


def main(argv):
  if len(argv) != 3:
    print 'Usage: gen_input_methods.py [input_methods.txt] [output]'
    sys.exit(1)
  login_xkb_layout_ids = []
  for line in fileinput.input(sys.argv[1]):
    line = line.strip()
    if not line or re.match(r'#', line):
      continue
    columns = line.split()
    assert len(columns) == 4 or len(columns) == 5, "Invalid format: " + line
    if len(columns) == 5:
      assert columns[4] == "login", "Invalid attribute: " + columns[4]
      login_xkb_layout_ids.append(columns[1])

  output = CreateEngineHeader(login_xkb_layout_ids)
  output_file = open(sys.argv[2], 'w')
  output_file.write(output)


if __name__ == '__main__':
  main(sys.argv)
