# Genetic
# Copyright (C) 2001 Jean-Baptiste LAMY
#
# This program is free software. See README or LICENSE for the license terms.

"""genetic.prog -- module for genetic programming
"""

import operator, random, UserList, copy
from genetic import organism


class LispList(UserList.UserList):
  """LispList implements a callable list, similar to Lisp one !
E.g.: LispList([func, a, b])() is the same that func(a, b).
"""
  def __call__(self, **args):
    def _subcall(object):
      if isinstance(object, LispList) or isinstance(object, Argument): return object(**args)
      return object
    
    l = map(_subcall, self)
    
    return l[0](*l[1:])
  
  def __repr__(self):
    if self[0] is operator.add: return "[%s]" % " + ".join(map(repr, self.data[1:]))
    if self[0] is operator.sub: return "[%s]" % " - ".join(map(repr, self.data[1:]))
    if self[0] is operator.mul: return "[%s]" % " * ".join(map(repr, self.data[1:]))
    if self[0] is operator.div: return "[%s]" % " / ".join(map(repr, self.data[1:]))
    return "%s%s" % (self[0].func_name, tuple(self[1:]))
  

class Argument:
  def __init__(self, name):
    self.name  = name
    
  def __copy__    (self      ): return self # Non changeable => useless to copy
  def __deepcopy__(self, dict): return self
  
  def __call__ (self, **args): return args[self.name]
  
  def __repr__(self): return self.name
  

class EquationContext:
  def __init__(self, **args):
    self.args = []
    for name, value in args.items():
      arg = Argument(name)
      self.args.append(arg)
      
    context = self # for access in inner class.
    class EquationChromosom(LispList, organism.Chromosom):
      DEFAULT_GENES = {
        "__change__"    :  0.5 ,
        "__mutation__"  :  0.6 ,
        "__mutampl__"   :100.0 ,
        "__mutsign__"   :  0.0 ,
        }
      def __init__(self, **args):
        LispList.__init__(self)
        organism.Chromosom.__init__(self, **args)
        
        if len(self.data) == 0:
          eq_parts  = []
          self.args = []
          for arg in context.args:
            # eq_parts = [ [* constant0 arg0], [* constant1 arg1], ...]
            eq_parts.append(EquationChromosom(data = [operator.mul, random.random() - 0.5, arg]))
            
          # equation = [+ [* constant0 arg0], [+ [* constant1 arg1], ...]]
          # NB: we cannot do [+ a, b, c, d ...]; operator.add accepts only 2 arguments.
          eq = reduce(lambda eq, eq_part: EquationChromosom(data = [operator.add, eq_part, eq]), eq_parts)
          self.data = [operator.add, eq, (random.random() - 0.5) * self.__mutampl__]
          
      def useless(self): return 0
      
      def crossover(self, other):
        pass
      
      def checkmutate(self):
        self.__mutation__ = .5
        #self.__mutampl__  = 50.0
        self.__change__   = .2
        
        newdict = self.checkmutate_object(self.__dict__)
        if not newdict is self.__dict__: mutated = self.__class__(**newdict)
        else:                            mutated = self
        
        #mutated = organism.Chromosom.checkmutate(self)
        
        #for i in range(1, len(self.data)):
        #  if random.random() < self.__change__:
        #    if mutated is self: mutated = self.__class__(self.__dict__)
        #    mutated.data[i] = mutated.new_equation_member()
            
        for i in range(1, len(self.data)):
          if random.random() < self.__change__:
            if mutated is self: mutated = self.__class__(self.__dict__)
            
            h = random.random()
            if   h < .6: mutated.data[i] = self.new_equation_member()
            elif h < .7: mutated.data[i] = EquationChromosom(data = [operator.add, mutated.data[i], self.new_equation_member()])
            elif h < .8: mutated.data[i] = EquationChromosom(data = [operator.sub, mutated.data[i], self.new_equation_member()])
            elif h < .9: mutated.data[i] = EquationChromosom(data = [operator.mul, mutated.data[i], self.new_equation_member()])
            else:        mutated.data[i] = EquationChromosom(data = [operator.div, mutated.data[i], self.new_equation_member()])
            
        return mutated
      
      def new_equation_member(self):
        h = random.random()
        if h < 0.3 : return (random.random() - 0.5) * self.__mutampl__
        if h < 0.35: return EquationChromosom(data = [operator.add, self.new_equation_member(), self.new_equation_member()])
        if h < 0.4 : return EquationChromosom(data = [operator.sub, self.new_equation_member(), self.new_equation_member()])
        if h < 0.45: return EquationChromosom(data = [operator.mul, self.new_equation_member(), self.new_equation_member()])
        if h < 0.5 : return EquationChromosom(data = [operator.div, self.new_equation_member(), self.new_equation_member()])
        #return random.choice(context.args)
        return EquationChromosom(data = [operator.mul, (random.random() - 0.5), random.choice(context.args)])
      
      def __repr__(self):
        return LispList.__repr__(self)
      
    self.Chromosom = self.__call__ = EquationChromosom

